<?php
/**
 * Zend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @category   Zend
 * @package    Zend_Validate
 * @copyright  Copyright (c) 2007 Boris Tomic <boris@kodmasin.com>
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 */
$my_dir = dirname(__FILE__);
require_once 'Zend/Validate/Interface.php';
require_once 'Zend/Validate/Exception.php';
require_once 'Zend/Filter/Interface.php';
require_once $my_dir . '/Array/Required.php';
require_once $my_dir . '/Array/Field.php';
/**
 * @category   Zend
 * @package    Zend_Validate
 * @copyright  Copyright (c) 2007 Boris Tomic <boris@kodmasin.net>
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 */
class Zend_Validate_Array implements Zend_Validate_Interface 
{
    /**
     * allowed names
     *
     * @var array
     */
    protected $_names;
    /**
     * filters
     *
     * @var array
     */
    protected $_filters;
    /**
     * is value validated
     *
     * @var boolean
     */
    protected $_validated;
    /**
     * validates
     *
     * @var array
     */
    protected $_validators;
    /**
     * array of values to be validated
     *
     * @var array
     */
    protected $_values;
    /**
     * errors if not valid
     *
     * @var array
     */
    protected $_user_errors;
    /**
     * valid values
     *
     * @var array
     */
    protected $_validatedValues;
    /**
     * messages for developer if invalid
     *
     * @var array
     */
    protected $_messages;
    /**
     * error codes for developer if invalid
     *
     * @var array
     */
    protected $_errors;
    /**
     * holds information is key required or not
     *
     * @var array
     */
    protected $_required;
    /**
     * sets validator available keys which will be validated
     *
     * @param array $names
     */
    public function __construct(array $names)
    {
        $this->_names = $names;
        $this->_filters = array();
        $this->_validated = false;
        $this->_validators = array();
        $this->_values = null;
        $this->_user_errors = array();
        $this->_validatedValues = array();
        $this->_messages = array();
        $this->_errors = array();
        $this->_required = array();
        //set all keys to not required
        foreach($this->_names as $name){
        	$this->_required[$name] = false;
        }
    }
    protected function _addFilter($name, Zend_Filter_Interface $filter){
    	$this->_isValidName($name);
        if(!isset($this->_filters[$name]))
            $this->_filters[$name] = array();
        $this->_filters[$name][] = $filter;
    }
    /**
     * add filter for key in array to be validated
     *
     * @param string $name name of key for which filter will be added
     * @param Zend_Filter_Interface $filter
     * @return Zend_Validate_Array this
     */
    public function &addFilter($name, Zend_Filter_Interface $filter)
    {
        if(is_array($name)){
        	foreach($name as $na){
        		$this->_addFilter($na, $filter);
        	}
        }
        else if($name == "*"){
        	foreach($this->_names as $kname){
        		$this->_addFilter($kname, $filter);
        	}
        }
        else{
        	$this->_addFilter($name, $filter);
        }
        $this->_reset();
        return $this;
    }
    
    protected function _addValidator($name, Zend_Validate_Interface $validate, $error = null, $options = null){
    	$this->_isValidName($name);
        if(!isset($this->_validators[$name]))
            $this->_validators[$name] = array();
        if($validate instanceof Zend_Validate_Array_Required){
        	$this->_required[$name] = true;
        }
        $this->_validators[$name][] = array($validate, $error, $options);
    }
    /**
     * add filter for key in array to be validated
     *
     * @param string $name name of key for which validator will be added
     * @param Zend_Validate_Interface $validate
     * @param string $error error message whichcan be displayed to end user
     * @return Zend_Validate_Array this
     */
    public function &addValidator($name, Zend_Validate_Interface $validate, $error = null, $options = null)
    {
    	/*
    	 * if $name is array of key names then add this validator to all keys menationed
    	 * in $name array
    	 */ 
    	if(is_array($name)){
        	foreach($name as $na){
        		$val_error = null;
        		$val_options = null;
        		if(isset($error[$na])){
        			$val_error = $error[$na];
        		}
        		else if(is_string($error)){
        			$val_error = $error . " (" . $na . ")";
        		}
        		if(isset($options[$na])){
        			$val_options = $options[$na];
        		}
        		$this->_addValidator($na, $validate, $val_error, $val_options);
        	}
        }
        //add this validator to all keys
        else if($name == "*"){
        	foreach($this->_names as $kname){
        		if(is_string($error)){
        			$this->_addValidator($kname, $validate, $error . " (" . $kname . ")", $options);
        		}
        		else{
        			$this->_addValidator($kname, $validate, $options);
        		}
        	}
        }
        //add validator to one key under name $name string
        else{
        	$this->_addValidator($name, $validate, $error, $options);
        }
        $this->_reset();
        return $this;
    }
    /**
     * sets array to be validated.
     *
     * @param array $values
     * @return Zend_Validate_Array this
     */
    protected function _setArray($values)
    {
		if(is_array($values)){
        	$this->_values = array();
        	foreach($this->_names as $name){
        		if(isset($values[$name]))
            		$this->_values[$name] = $values[$name];
            	else
            		$this->_values[$name] = null;
        	}
        	$this->_reset();
		}
        return $this;
    }
    /**
     * clears all values set by validate function. it does not clear already set filters
     * and validators
     *
     */
    private function _reset()
    {
        $this->_validated = false;
        $this->_messages = array();
        $this->_validatedValues = array();
        $this->_user_errors = array();
        $this->_errors = array();
    }
    /**
     * check if key is set to be checked
     *
     * @param string $name name of key to be chacked
     */
    private function _isValidName($name)
    {
        if(!in_array($name, $this->_names)){
            return false;
        }
        return true;
    }
    /**
     * all 'clever' work of this validator is done here. First all filters are applied to
     * apropriate values and then this filtered values are validated. If any validator fall
     * this function returns false.
     *
     * @return boolean true if array is valid and false otherwise
     */
    public function validate()
    {
        if($this->_values === null){
            throw new Zend_Validate_Exception("Could not validate because there are no data to validate.");
        }
        foreach($this->_names as $name){
            $this->_validatedValues[$name] = $this->_values[$name];
            if(isset($this->_filters[$name])){
                foreach($this->_filters[$name] as $filter){
                    $this->_validatedValues[$name] = $filter->filter($this->_validatedValues[$name]);
                }
            }
            if(isset($this->_validators[$name]) && (!$this->_isEmpty($this->_validatedValues[$name]) || $this->_required[$name])){
                $notBreakCain = true;
            	foreach($this->_validators[$name] as $validator){
                    if(!$validator[0]->isValid($this->_validatedValues[$name]) && $notBreakCain){
                        if(!isset($this->_user_errors[$name]))
                            $this->_user_errors[$name] = array();
                        if(!isset($this->_errors[$name]))
                        	$this->_errors[$name] = array();
                        if(isset($validator[2]['breakChain']) && $validator[2]['breakChain']){
                        	$notBreakCain = false;
                        }
                        $this->_user_errors[$name][] = $validator[1];
                        $this->_messages[$name][] = $validator[0]->getMessages();
                        $this->_errors[$name][] = $validator[0]->getErrors();
                    }
                }
            }
        }
        $this->_validated = true;
        return $this->_isValid();
    }
    private function _isEmpty($value){
    	if($value == null || $value == "")
    		return true;
    	return false;
    }
    /**
     * returns errors of all keys of for one key if its name supplied.
     *
     * @return mixed array of errors sorted by key names or if $name supplied just errors
     * for key with name $name.
     */
    public function getUserMessages($name = null)
    {
        if(!$this->_validated)
            $this->validate();
        if($name !== null)
        	if($this->_isValidName($name))
        		return $this->_user_errors[$name];
        return $this->_user_errors;
    }
    /**
     * check if some valu is valid
     *
     * @param string $name key name
     * @return boolean true if valid false if not
     */
    public function isKeyValid($name)
    {
        if($this->_isValidName($name))
        	return false;
        if(!$this->_validated)
            $this->validate();
        if(count($this->_errors) > 0 && array_key_exists($name, $this->_errors))
            return false;
        return true;
    }
    /**
     * internal function which checks if wanted array is valid
     *
     * @return boolean true if valid false if not
     */
    protected function _isValid()
    {
        if(!$this->_validated)
            $this->validate();
        if(count($this->_user_errors) == 0)
            return true;
        return false;
    }
    /**
     * validate array
     *
     * @param array $values
     * @return boolean true if valid false if not
     */
    public function isValid($value)
    {
        $this->_setArray($value);
        return $this->validate();
    }
    /**
     * returns valid value of wanted key or null if is not valid
     *
     * @param string $key name of key for wich validated value will be returnned. If null all valid values will be returned.
     * @return mixed valid value if valid or null if not valid
     */
    public function getValidValue($key = null)
    {
    	if(!$this->_validated)
            $this->validate();
        if($key === null)
        	return $this->_validatedValues;
        if($this->_isValidName($key)){
            return $this->_validatedValues[$key];
        }
        return null;
    }
    /**
     * returns original error messages for invalid validators
     *
     * @return array messages
     */
    public function getMessages()
    {
        return $this->_messages;
    }
    /**
     * Add user error for key with $name
     *
     * @param string $name name of key to wich we will add errror
     * @param string $error error
     */
    public function addUserErrorMessage($name, $error){
        if(!isset($this->_user_errors[$name]))
            $this->_user_errors[$name] = array();
        $this->_user_errors[$name][] = $error;
    }
    /**
     * Return error codes
     *
     * @return array with error codes
     */
    public function getErrors(){
    	return $this->_errors;
    }
    /**
     * Returns value which does not have to be valid.
     *
     * @param string $key
     * @return mixed
     */
    public function getValues($key = null){
    	if(count($this->_values) > 0){
    		if($key === null){
        		return $this->_values;
    		}
        	if($this->_isValidName($key))
            	return $this->_values[$key];
    	}
        return null;
    }
    public function __get($name){
    	return $this->getValidValue($name);
    }
    /**
     * returns field object so that you can set validators and
     * filters for it
     *
     * @param string $name naem of field (key)
     * @return Zend_Validate_Array_Field or null if $name is not valid key
     */
    public function getField($name){
    	if($this->_isValidName($name)){
    		return new Zend_Validate_Array_Field($name, $this);
    	}
    	return ;
    }
}